function cell_loc_array = findCellsFolder
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           findCellsFolder.m                             %
                              Ver = '0.01';
%                      Apr. 13, 2013 by Ippei Kotera                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This function finds xy coordiantes of bacterial cells in all the images 
% in a specified folder. The calculation can be accelerated by parallel 
% computing by assigning image slices to multiple CPU cores.
%
% v0.01; Initial release

close all;

disp(['Running findCellsFoler Ver. ', num2str(Ver), ' ...']);

flgParallel = 0; % Put 1 for parallel computation.
flgShowPics = 0; % Put 1 for image display. Only for non-parallel computation.
flgBackSubtract = 0; % Background subtraction of images. Might be more accurate but slower.
numCores = 8; % Specify number of CPU cores
numSlices = numCores;
depthBit = 8; % Color depth of images
numFolders = 1; 
nameFolder{1} = [];
hF = nan;

% Chech number availble CPU cores
numCoresAvailable = str2double(getenv('NUMBER_OF_PROCESSORS'));
if numCores > numCoresAvailable
   numCores = numCoresAvailable; %#ok<NASGU>
end

nameFolder{numFolders} =...
   'C:\SiMing Data\Feb 25\2326_feb25_f';

% Start matlab workers for parallel computation
if flgParallel
   matlabpool close force local; %#ok<UNRCH>
   if (matlabpool('size')) < numCores
      matlabpool('local', numCores);
   end
end

nameFolderSave = nameFolder{numFolders,1};
display(nameFolder{numFolders,1});

% Parameters for cell recognition
minCellIntensity(numFolders) = 7 / (2 ^ depthBit - 1);
minCellSize(numFolders) = 5;
maxEccentricity(numFolders) = 0.95;
maxHaloSize(numFolders) = 75;

cell_loc_array = cell(numFolders,1);

for countFolder = 1:numFolders
   folderCurrent = nameFolder{countFolder, 1};
   cd(folderCurrent)
   nameImages = dir('*.jpg');
   numImages = size(nameImages,1);
%    cell_loc = cell(numImages,1);
   
   tic;
   
   % Divide images into slices
   numInSlice = ceil(numImages / numSlices);
%    numInLastSlice = numImages - numInSlice * numSlices - 1;
   
   if flgParallel
      % For parallel computation
      parfor sl = 1:numSlices %#ok<UNRCH>
         
         if sl <= numCores
            pause(0.3 * sl);
         end
         
         % Separate loop section for parallel computing
         [cellTemp, ~, ~] = nLoop(sl, numImages, numInSlice,...
            numFolders, countFolder, nameImages,...
            minCellIntensity, minCellSize, maxEccentricity,...
            maxHaloSize, flgBackSubtract);
         
         cellTempTemp(:, sl) = cellTemp;
         
      end
   else
      % For non-parallel computation
      for sl = 1:numSlices
         
         [cellTemp, im, imBW] = nLoop(sl, numImages, numInSlice,...
            numFolders, countFolder, nameImages,...
            minCellIntensity, minCellSize, maxEccentricity,...
            maxHaloSize, flgBackSubtract);
         
         cellTempTemp(:, sl) = cellTemp; %#ok<AGROW>
         
         % For image display
         if flgShowPics == 1
            if isnan(hF)
               hF = figure('Units', 'pixels', 'Position', [100 30 1000 660]);
            end
            im = imrotate(im, 180);
            
            % Create axis for image1 display
            hA1 = axes('Parent', hF, 'Units', 'pixels',...
               'Position', [10 10 480 640], 'Layer','top'); %#ok<*LAXES>
            hA2 = axes('Parent', hF, 'Units', 'pixels',...
               'Position', [510 10 480 640], 'Layer','top');
            imagesc(im, 'Parent', hA1);
            imagesc(imBW, 'Parent', hA2);
            
            % Get rid of the tick bars
            set(hA1, 'XTick', [], 'YTick', []);
            daspect([1 1 1]);
            set(hA2, 'XTick', [], 'YTick', []);
            
            drawnow;

         end
         
      end
   end
   
   cell_loc = reshape(cellTempTemp, numInSlice * numSlices, 1);
   
   % Get rid of cell rows that are all empty (vectorized)
   cell_loc(all(cellfun(@isempty, cell_loc), 2), :) = [];
   
   toc;
   
   nameSave = strcat(nameFolderSave, '\cell_loc.mat');
   
   save(nameSave, 'cell_loc');
   
   cell_loc_array{countFolder} = cell_loc;
   
end

end


function [cellTemp, im, imBW] = nLoop(sl, numImages, numInSlice,...
   numFolders, countFolder, nameImages,...
   minCellIntensity, minCellSize, maxHaloSize,...
   maxEccentricity, flgBackSubtract)

% Preallocate the cells
cellTemp = cell(numInSlice, 1);

for cs = 1:numInSlice
   
   % For the last slice, if the current image exceeds the total
   %number of images then, break the loop.
   ct = (sl - 1) * numInSlice + cs;
   if (ct > numImages)
      break;
   end
   nameIm = nameImages(ct, 1).name;

   % FindCells function
   [temp, im, imBW] = findCells(nameIm,...
      minCellIntensity(countFolder),...
      minCellSize(countFolder),...
      maxHaloSize(numFolders),...
      maxEccentricity(numFolders),...
      flgBackSubtract);
   
   [cellTemp(cs, 1)] = {temp};
   
   disp(ct);

end

end


function [cell_loc, im, imBW]  =...
   findCells(nameIm, minCellIntensity, minCellSize, maxHaloSize,...
   maxEccentricity, flgBackSubtract)

im = imread(nameIm);

se1 = strel('disk', 1); %for first erode
se2 = strel('disk', 5); %for dilation
if flgBackSubtract
   im = uint8(bpass(im, 1, 20)); % Band-pass filter to remove background
end
imBW = im2bw(im, minCellIntensity); % matlab's thresholding function.
imBW = imerode(imBW, se1);
imBW = bwareaopen(imBW, minCellSize); % Remove cells that are smaller that thresh
imBW = imdilate(imBW, se1);
imBW = imdilate(imBW, se2);
imBW = imfill(imBW, 'holes');
imBW = imerode(imBW, se2);
imBW = imrotate(imBW, 180);     % ADDED 04/25/13 BY SMZ

structCell = regionprops(imBW, 'Centroid', 'Area', 'Eccentricity');
markDeletion = zeros(length(structCell),1);

% Mark cells for deletion if they are too large or eccentric
for countRegion = 1 : length(structCell)
   if (structCell(countRegion).Area > maxHaloSize) &&...
         (structCell(countRegion).Eccentricity > maxEccentricity)
      markDeletion(countRegion) = 1;
   end
end

cell_loc = struct2cell(structCell)';
cell_loc = cell2mat(cell_loc);
cell_loc = circshift(cell_loc,[0 -1]);
cell_loc(markDeletion == 1,:) = [];

if isempty(cell_loc)
   cell_loc = [NaN,NaN,NaN,NaN];
end

cell_loc(1:size(cell_loc,1),5)= NaN;

% cell_loc has x-coordinates, y-coordinates, eccentricity, and area

end



